% finaquant Financial Analytics - www.finaquant.com
% Copyright: Tunc Ali Ktkcoglu 2012, version: 24April2012
% TITLE: Share Class Fee Analyzer
% Related web page:
% http://finaquant.com/fee-calculator-and-analyzer-for-share-classes-of-mutual-funds-1-calculation/991
%
% Compare total returns of different share classes of a mutual fund
% Share classes:
% A: Front-end load with volume discounts, low annual distribution fee
% B: Back-end load (CDSC), medium-high distribution fee
% C: Low back-end load (CDSC), high distribution fee

%**************************************************************************
% Share class specific input parameters
%**************************************************************************
disp('Set share class specific input parameters');
% I try to define parameters in an ordered way here (array of structure) to
% avoid a mess of parameter names

clear InputParameters
% Share class name
InputParameters(1).ShareClass = 'A';
InputParameters(2).ShareClass = 'B';
InputParameters(3).ShareClass = 'C';

% Front-end load rates
InputParameters(1).FrontEndLoadRates = [       % percentage and exclusive upper limit
    5.00        50000;      
    4.50        100000;     
    3.70        250000;
    2.80        500000;
    2.00        999999999];

InputParameters(2).FrontEndLoadRates = [0  999999999];  % B
InputParameters(3).FrontEndLoadRates = [0  999999999];  % C

% Annual distribution fee rates in percentage
InputParameters(1).DistrFeeRate = 0.2;  
InputParameters(2).DistrFeeRate = 0.8;
InputParameters(3).DistrFeeRate = 1.0; 

% Back-end load with CDSC
InputParameters(1).BackEndLoadCDSC = 0;
InputParameters(2).BackEndLoadCDSC = [5 4 3 2 1];
InputParameters(3).BackEndLoadCDSC = [1  1];

%**************************************************************************
% Common fixed parameters for all share classes
%**************************************************************************
disp('Set fixed common input parameters');
AnnualManagementFeeRates = [   
    80          50000;         
    70          100000;        
    60          999999999];
InputParameters(1).MngFeeRates = AnnualManagementFeeRates;
InputParameters(2).MngFeeRates = AnnualManagementFeeRates;
InputParameters(3).MngFeeRates = AnnualManagementFeeRates;

MngFeeScaleLogic = 'class';
InputParameters(1).MngFeeScaleLogic = MngFeeScaleLogic;
InputParameters(2).MngFeeScaleLogic = MngFeeScaleLogic;
InputParameters(3).MngFeeScaleLogic = MngFeeScaleLogic;

OtherAnnualServiceFeeRates = [10 0];    % 10 BP + $0 per share, annually
InputParameters(1).ServiceFeeRates = OtherAnnualServiceFeeRates;
InputParameters(2).ServiceFeeRates = OtherAnnualServiceFeeRates;
InputParameters(3).ServiceFeeRates = OtherAnnualServiceFeeRates;

OtherSubscriptionFeeRates = [0 0];      % 0% + $0 per share, at subscription
InputParameters(1).SubsFeeRates = OtherSubscriptionFeeRates;
InputParameters(2).SubsFeeRates = OtherSubscriptionFeeRates;
InputParameters(3).SubsFeeRates = OtherSubscriptionFeeRates;

OtherRedemptionFeeRates = [0 0];        % 0% + $0 per share, at redemption
InputParameters(1).RedempFeeRates = OtherRedemptionFeeRates;
InputParameters(2).RedempFeeRates = OtherRedemptionFeeRates;
InputParameters(3).RedempFeeRates = OtherRedemptionFeeRates;

%**************************************************************************
% Simulation of fee calculation with variable parameters
%**************************************************************************
disp('Simulation of fee calculation with variable parameters');
% Varied parameters:
% Share classes
% NumberOfShares
% N: Total number of calculation periods
% AssumedFixedReturn: Assumed fixed fund return for all calculation periods

% Simulation parameters
ShareSamples = 10;
Nmax = 10;
MaxReturn = 10;
SharePrice = 500;
MaxShareInd = 10;
ShareMultiplier = 100;  % i.e. #shares = 100, 200, ..., 1000

% call simultion function
[TotalFundReturn, FinalAssetValue] = SimulateTotalReturns( ...
    InputParameters,    ...
    Nmax,               ... % upper limit to periods for simulation (1,2, ... Nmax)
    MaxReturn,          ... % upper limit to assumed fixed returns (1,2,... MaxReturn)
    SharePrice,         ... % initial purchase price of a share
    MaxShareInd,        ... % #shares = [1,2,... MaxShareInd] * ShareMultiplier
    ShareMultiplier);
         
%**************************************************************************
% Analyze results to compare total returns of share classes
%**************************************************************************
% assumed fixed fund return for all 3 cases below
AssumedFixedReturn = 5;

% CASE 1
disp('Case 1: Assumed annual return = 5%, #shares = 100, N is varied')
NumberOfShares = 100;
IndShares = NumberOfShares / ShareMultiplier;
for i=1:Nmax
    TotalReturnsA(i) = TotalFundReturn(1,IndShares,i,AssumedFixedReturn);
    TotalReturnsB(i) = TotalFundReturn(2,IndShares,i,AssumedFixedReturn);
    TotalReturnsC(i) = TotalFundReturn(3,IndShares,i,AssumedFixedReturn);
end
x = 1:Nmax;
p = plot(x,TotalReturnsA,'b',x,TotalReturnsB,'g',x,TotalReturnsC,'r');
set(p,'LineWidth',2);
xlabel('period N (years)')
ylabel('total return in %')
title('Assumed annual return = 5%, #shares = 100 (A blue, B green, C red)')
figure

% CASE 2
disp('Case 2: Assumed annual return = 5%, #shares = 500, N is varied')
NumberOfShares = 500;
IndShares = NumberOfShares / ShareMultiplier;
for i=1:Nmax
    TotalReturnsA(i) = TotalFundReturn(1,IndShares,i,AssumedFixedReturn);
    TotalReturnsB(i) = TotalFundReturn(2,IndShares,i,AssumedFixedReturn);
    TotalReturnsC(i) = TotalFundReturn(3,IndShares,i,AssumedFixedReturn);
end
x = 1:Nmax;
p = plot(x,TotalReturnsA,'b',x,TotalReturnsB,'g',x,TotalReturnsC,'r');
set(p,'LineWidth',2);
xlabel('period N (years)');
ylabel('total return in %');
title('Assumed annual return = 5%, #shares = 500 (A blue, B green, C red)');
figure

% CASE 3
disp('Case 3: Assumed annual return = 5%, #shares = 1000, N is varied')
NumberOfShares = 1000;
IndShares = NumberOfShares / ShareMultiplier;
for i=1:Nmax
    TotalReturnsA(i) = TotalFundReturn(1,IndShares,i,AssumedFixedReturn);
    TotalReturnsB(i) = TotalFundReturn(2,IndShares,i,AssumedFixedReturn);
    TotalReturnsC(i) = TotalFundReturn(3,IndShares,i,AssumedFixedReturn);
end
x = 1:Nmax;
p = plot(x,TotalReturnsA,'b',x,TotalReturnsB,'g',x,TotalReturnsC,'r');
set(p,'LineWidth',2);
xlabel('period N (years)');
ylabel('total return in %');
title('Assumed annual return = 5%, #shares = 1000 (A blue, B green, C red)');
figure

%**************************************************************************
% Surface plot for optimal share class
%**************************************************************************
disp('Surface plot for optimal share class: Assumed annual return = 5%')
AssumedFixedReturn = 5;

[IndShares,Period] = meshgrid(1:10, 1:10);
[rows, cols] = size(IndShares);
OptimalShareClass = zeros(rows, cols);
for i=1:rows
    for j=1:cols
        OptimalShareClass(i,j) = FindOptimalShareClass(TotalFundReturn,IndShares(i,j),Period(i,j),AssumedFixedReturn);
    end
end
surf(IndShares,Period,OptimalShareClass);
axis([1 10 1 10 1 3]);
view([0 0 1]);
xlabel('# shares / 100');
ylabel('# periods (N)');
title('Optimal share class (return = 5%, A blue, B green, C red)');
% colorbar
figure

disp('Surface plot for optimal share class: Assumed annual return = 10%')
AssumedFixedReturn = 10;

[IndShares,Period] = meshgrid(1:10, 1:10);
[rows, cols] = size(IndShares);
OptimalShareClass = zeros(rows, cols);
for i=1:rows
    for j=1:cols
        OptimalShareClass(i,j) = FindOptimalShareClass(TotalFundReturn,IndShares(i,j),Period(i,j),AssumedFixedReturn);
    end
end
surf(IndShares,Period,OptimalShareClass);
axis([1 10 1 10 1 3]);
view([0 0 1]);
xlabel('# shares / 100');
ylabel('# periods (N)');
title('Optimal share class (return = 10%, A blue, B green, C red)');
% colorbar
